<!--
	The generic cropper tag

	Calling convention:

	window.ptvbus.trigger('crop-image', {
		imgSrc:         urlOrFile,                 // required
		height:         heightInPx,                // optional
		width:          widthInPx,                 // optional
		rounded:        roundedCropper,            // optional
		bgClass:        bgColorClass,              // optional
		mime:           mimeType,                  // optional
		maxOutputScale: outputScale,               // optional
		cropCallback:   function(data, blob, done) // required
	});

	urlOrFile can be either an image URI or a File object.

	cropCallback should handle data (the image as a base64 data url),
	the blob (the image as a blob object, good for ajax file uploads),
	and call the done argument as a function to close the modal
	by calling done()

	mime must be an mimetype. Example:
	"image/png"
	"image/jpeg"

	maxOutputScale is the maximum output scale of the image.
	This is used to allow for cleaner scaling down server-side.
	Default value is 3

	Example:

	window.ptvbus.trigger('crop-image', {
		imgSrc:  "/images/mascott/mimi_cat.png",
		height:  500,
		width:   500,
		bgClass: "bg-dark",
		cropCallback: function(data, blob, done) {
			$.post("/handler/imageupload", {
				img: image
			}).always(done)
		}
	});

	The .always can also be
	.always(function() {
		done();
	})
	if you need to do additional processing in there.

	WARNING: Putting done() inside the success callback will not always give correct behaviour.
	ALWAYS put it somewhere where it will always be called.
-->

<modal-cropper class="modal fade cropper-modal-container" tabindex="-1" role="dialog" aria-labelledby="modal-cropper">
	<div class="modal-dialog modal-lg" role="document" ref="modalContainer">
		<div class="modal-content">
			<div class="modal-header">
				<h5 class="modal-title" data-i18n="cropper.title">Crop image</h5>
				<button type="button" class="close" aria-label="Close" onclick={closeModal}>
					<span aria-hidden="true">&times;</span>
				</button>
			</div>
			<div class="modal-body">
				<div class="d-none {this.bgClass}" ref="bgTarget"></div>
				<div class="text-center">
					<div class="cropper-wrap {cropper-rounded: rounded} d-inline-block" style="background-color: {bgColor}; width: {visualSize.width}px; height: {visualSize.height}px">
						<!--  crossOrigin required for being able to generate a data URL from external sources  -->
						<img ref="img" class="mw-100" src={imgSrc} crossOrigin="Anonymous">
					</div>
					<div class="text-muted mt-2">
						<virtual if={mobileDevice} data-i18n="cropper.tip_mobile">Tap & drag to move, pinch to zoom</virtual>
						<virtual if={!mobileDevice} data-i18n="cropper.tip_desktop">Click & drag to move, scroll to zoom</virtual>
					</div>
				</div>
			</div>
			<div class="modal-footer justify-content-center">
				<button class="btn btn-primary" onclick={save} hide={finishing} data-i18n="cropper.save">Save</button>
				<button class="btn " show={finishing}><i class="far fa-fw fa-spinner fa-spin"></i></button>
			</div>
		</div>
	</div>

	<script>
		this.rounded = false;
		this.imgSrc = "";
		this.cropperSrc = "";
		this.bgClass = "bg-dark";
		this.bgColor = "";
		this.finishing = false;
		// Will be passed onto server to scale
		// Rule of thumb: Set to 3x target size before sending to the server
		this.outputSize = {
			height: 200,
			width: 200
		};
		this.visualSize = {
			height: 200,
			width: 200
		};
		this.visualScale = 1;
		this.outputScale = 3;
		this.currCrop = {
			x: 0,
			y: 0,
			width: 0,
			height: 0
		};
		/*
			Try:
			image/jpeg
			image/png
		*/
		this.mime = "image/png";
		this.cropCallback = null;

		this.mobileDevice = /Android|BlackBerry|iPhone|iPod|iPad|Opera Mini|IEMobile|WPDesktop/.test(window.navigator.userAgent);

		if (!HTMLCanvasElement.prototype.toBlob) {
			Object.defineProperty(HTMLCanvasElement.prototype, 'toBlob', {
				value: function (callback, type, quality) {
					var canvas = this;
					setTimeout(function() {
						var binStr = atob( canvas.toDataURL(type, quality).split(',')[1] ),
							len = binStr.length,
							arr = new Uint8Array(len);

						for (var i = 0; i < len; i++ ) {
							arr[i] = binStr.charCodeAt(i);
						}

						callback( new Blob( [arr], {type: type || 'image/png'} ) );
					});
				}
			});
		}

		// Get the visual cropper bounds corrected to the container size
		getCropperBounds(width, height) {

			// A bit of a cheat, but it works
			var elem = document.createElement("div");
			elem.className = "modal-dialog modal-lg";
			document.body.appendChild(elem);
			var containerStyle = getComputedStyle(elem);
			var bounds = {
				width: parseFloat(containerStyle.width) - (parseFloat(containerStyle.fontSize) * 2),
				height: height
			};
			var scale = 1;
			elem.remove();

			var inputRation = width/height;
			if (bounds.width < width) {
				scale = bounds.width / width;
				bounds.height = height * scale;
			} else {
				bounds.width = width;
			}


			return {
				bounds: bounds,
				scale:  scale
			};
		}

		// Close rhe overlay
		closeModal() {
			$(this.root).modal('hide');
		}

		// Update crop data
		crop(evt) {
			var data = evt.detail;
			this.currCrop = data;
		}

		// Initialize/re-initialize the cropper based on internal properties/settings
		initCropper() {
			if (this.cropper) {
				this.cropper.destroy();
			}
			this.cropper = new Cropper(this.refs["img"], {
				viewMode:           3,
				//aspectRatio:        this.visualSize.width/this.visualSize.height,
				//responsive:         true,
				//guides:             false,
				highlight:          false,
				center:             false,
				rotatable:          true,
				scalable:           true,
				background:         false,
				minContainerWidth:  this.visualSize.width,
				minContainerHeight: this.visualSize.height,

				autoCropArea:             0.5,
				//cropBoxMovable:           false,
				//cropBoxResizable:         false,
				toggleDragModeOnDblclick: false,
				dragMode:                 "move",
				//minCropBoxWidth:          this.visualSize.width,
				//minCropBoxHeight:         this.visualSize.height,
			});
		}

		// Figure out if we need to downscale the image
		getDownscale() {
			return {
				height: Math.min(this.currCrop.height, this.outputSize.height*this.outputScale),
				width:  Math.min(this.currCrop.width,  this.outputSize.width*this.outputScale)
			};
		}

		// Save the crop data, calling the provided callback
		save() {
			var outSize = this.getDownscale();

			var canvas = document.createElement('canvas');
			canvas.width = outSize.width;
			canvas.height = outSize.height;
			var ctx = canvas.getContext('2d');

			// Set background colour
			ctx.fillStyle = this.bgColor;
			ctx.fillRect(0, 0, canvas.width, canvas.height);

			// Add image over it
			ctx.drawImage(this.refs["img"], this.currCrop.x, this.currCrop.y, this.currCrop.width, this.currCrop.height, 0, 0, canvas.width, canvas.height);

			// Get the image data
			var url = canvas.toDataURL(this.mime);
			this.finishing = true;
			canvas.toBlob(function(blob) {
				this.cropCallback(url, blob, this.closeModal);
			}.bind(this), this.mime);
		}

		// Set up the cropper when an image loads
		imgLoad(evt) {
			if (this.refs["img"].naturalHeight === this.outputSize.height && this.refs["img"].naturalWidth === this.outputSize.width) {
				this.currCrop = {
					x:      0,
					y:      0,
					width:  this.outputSize.width,
					height: this.outputSize.height
				};

				this.save();
				return;
			}

			if (this.cropperSrc === this.imgSrc) {
				return;
			}
			this.cropperSrc = this.imgSrc;
			this.initCropper();
			$(this.root).modal({show: true, backdrop: 'static'});
		};

		imgError() {
			displayErrorMsgCustom("Invalid image format");
		};

		// Cropper "constructor" (called on ptvbus)
		window.ptvbus.on("crop-image", function(data) {
			var needsReader = data.imgSrc instanceof Blob || data.imgSrc instanceof File;

			this.rounded = data.rounded || false;

			if (!needsReader && typeof data.imgSrc !== "string") {
				throw "imgSrc must be a string, blob or file";
			}
			if (typeof data.cropCallback !== "function") {
				throw "cropCallback must be defined and be a function";
			}

			var processInput = function(data) {
				this.bgClass = data.bgClass || "bg-dark";
				this.cropCallback = data.cropCallback;
				this.cropperSrc = "";
				this.imgSrc = "";
				this.finishing = false;
				this.outputSize.height = data.height || 200;
				this.outputSize.width = data.width || 200;
				var visualBounds = this.getCropperBounds(this.outputSize.width, this.outputSize.height);
				this.visualSize = visualBounds.bounds;
				this.visualScale = visualBounds.scale;
				this.mime = data.mime || "image/png";
				this.outputScale = data.maxOutputScale || 3;

				this.update();

				this.bgColor = window.getComputedStyle(this.refs["bgTarget"]).backgroundColor;
				this.imgSrc = data.imgSrc;

				this.update();
			}.bind(this);

			if (needsReader) {
				var reader = new FileReader();
				reader.addEventListener("load", function() {
					processInput(Object.assign(data, {imgSrc: reader.result}));
				}.bind(this));
				reader.readAsDataURL(data.imgSrc);
			} else {
				processInput(data);
			}
		}.bind(this));

		// Resize breakpoint to handle resizing of the cropper
		window.ptvbus.on('bs-size-breakpoint', function() {
			if (!this.cropper) {
				return;
			}

			var visualBounds = this.getCropperBounds(this.outputSize.width, this.outputSize.height);
			if (visualBounds.scale === this.visualScale) {
				return;
			}

			this.visualSize = visualBounds.bounds;
			this.visualScale = visualBounds.scale;

			this.update();
			this.initCropper();
		}.bind(this))

		this.on('mount', function() {
			this.refs["img"].addEventListener('load', this.imgLoad);
			this.refs["img"].addEventListener('error', this.imgError);
			this.refs["img"].addEventListener('crop', this.crop);
			this.refs["img"].addEventListener('ready', this.cropReady);

			// Detect if someone is being naughty with the code
			$(this.root).on('show.bs.modal', function() {
				if (!this.cropper) {
					console.error("%cI know you did that. Don't do that. Use the ptvbus API.", "font-size: 32px");
				}
			}.bind(this));

			// Automatically destroy the cropper when the modal is hidden
			$(this.root).on('hidden.bs.modal', function() {
				if (this.cropper) {
					this.cropper.destroy();
					this.cropper = null;
				}
			}.bind(this));
		}.bind(this));
	</script>
</modal-cropper>